package gov.va.mhv.util.xml;

import java.io.IOException;
import java.io.StringReader;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.jdom2.Attribute;
import org.jdom2.Content;
import org.jdom2.Document;
import org.jdom2.Element;
import org.jdom2.JDOMException;
import org.jdom2.Namespace;
import org.jdom2.input.DOMBuilder;
import org.jdom2.input.SAXBuilder;
import org.jdom2.output.DOMOutputter;
import org.jdom2.output.Format;
import org.jdom2.output.XMLOutputter;

public class XmlHelper {
    private static Logger logger = Logger.getLogger(XmlHelper.class.getName());
    private static final DOMBuilder dombuilder = new DOMBuilder();
    
    private XmlHelper() {
    	
    }
    
    /**
     * Creates a new Element using the supplied name, value, and namespace.  The created element will be
     * added to the parent element.  If the value is null, the attribute will not be created or added to the parent.
     * @param parent
     * @param name
     * @param value
     * @param ns
     */
    public static void addElement(Element parent, String name, String value, Namespace ns) {
        if(value != null) {
            Element newEl = new Element(name, ns);
            newEl.addContent(value);
            parent.addContent(newEl);
        }
    }
    
    public static Element addElement(String name, Namespace ns) {
        return new Element(name, ns);
    }
    
    public static Element addElement(String name, String value, Namespace ns) {     
        Element newEl = new Element(name, ns);
        if (value != null) {
            newEl.addContent(value);
        }
        return newEl;
    }
    
    
    
    /**
     * Will take the given input string and will return the corresponding
     * Document object.  
     */
    public static Document parseString(final String input) throws JDOMException, IOException {
    	if(logger.isLoggable(Level.FINEST)) {
    		logger.finest("XML string to parse: \n" + input);
    	}
    	
    	SAXBuilder builder = new SAXBuilder();
    	builder.setFeature("http://apache.org/xml/features/nonvalidating/load-external-dtd", false);

    	Document newDoc = null;
    	newDoc = builder.build(new StringReader(input));
    	
    	return newDoc;
    }

    public static Document newDoc() {
        return new Document();
    }
    
    public static Document toJDOM(org.w3c.dom.Document w3cDocument) {
        return dombuilder.build(w3cDocument);
    }

    public static org.w3c.dom.Document toDOM(org.jdom2.Document jdomDoc) throws JDOMException {
    	DOMOutputter outputter = new DOMOutputter();
    	return outputter.output(jdomDoc);
    }

    public static org.w3c.dom.Element toDOM(org.jdom2.Element jdomElement) throws JDOMException {
    	return toDOM(new Document(jdomElement)).getDocumentElement();
    }

    

    
    /**
     * Returns the content of the supplied Document using the Format class' pretty format
     * and the newline character as the line separator.  The prolog is not omitted
     * @param doc The org.jdom.Document object.
     * @return The Document content as a String.
     */
    public static String getTextFromDocument(Document doc) {
        return getTextFromDocument(doc, false);
    }
    
    
    /**
     * Returns the content of the supplied Document using the Format class' pretty format
     * and the newline character as the line separator. The xml string will not have the 
     * xml declaration. It is common to omit this in uses such as SOAP and XML-RPC calls. 
     * @param doc The org.jdom.Document object.
     * @return The Document content as a String.
     */
    public static String getTextFromDocument(final Document doc, final boolean omit) {
        Format format = Format.getPrettyFormat();
    	format.setLineSeparator(System.getProperty("line.separator"));
    	format.setOmitEncoding(true);
        return getTextFromDocument(doc, format, omit);
    }
    
    public static String getTextFromDocument(final Document doc, final Format format) {
    	return getTextFromDocument(doc, format, false);
    }

    /**
     * Returns the content of the supplied Document using the supplied Format or a default
     * format if null is passed as the Format parameter.
     * @param doc The org.jdom.Document object.
     * @param format The org.jdom.output.Format object or null.
     * @return The Document content as a String.
     */
    public static String getTextFromDocument(final Document doc, Format format, final boolean omit) {
        XMLOutputter xmlString;
        if (format == null) {
        	format = Format.getRawFormat();
        }

        format.setOmitDeclaration(omit);
        xmlString = new XMLOutputter(format);

        String result = xmlString.outputString(doc);
        return result;
    }

    public static String getTextFromElement(Element element) {
        Format format = Format.getPrettyFormat();
        format.setLineSeparator(System.getProperty("line.separator"));
        return getTextFromElement(element, format);
    }
    
    public static String getTextFromElement(final Element element, Format format) {
    	XMLOutputter xmlString;
    	if (format == null) {
    		format = Format.getRawFormat();
    	}
    	xmlString = new XMLOutputter(format);

    	String result = xmlString.outputString(element);
    	return result;
    }
    

    public static Set<Namespace> getAllNamesspaces(final Element element) {
        Set<Namespace> result = new HashSet<Namespace>();
        result.add(element.getNamespace());
        
        Iterator<Content> it = element.getDescendants();
        while (it.hasNext()) {
            Content c = it.next();
            if (c instanceof Element) {
                result.add(((Element)c).getNamespace());
            }
        }
        return result;
    }
    
    /**
     * Will search the dom tree for all elements and attributes with the oldNM namespace
     * and if found will modify to the newNS namespace
     * @param doc
     * @param oldNS
     * @param newNS
     */
	public static void modifyNamespace(Element element, Namespace oldNS, Namespace newNS) {
        element.removeNamespaceDeclaration(oldNS);
    	
        //first check the attributes and change if needed
        List<Attribute> attributes = element.getAttributes();
        for(Attribute currAtt : attributes) {
            if(currAtt.getNamespaceURI().equals(oldNS.getURI())) {
                currAtt.setNamespace(newNS);
            }
        }
        
        //Now change the elements
        if(element.getNamespaceURI().equals(oldNS.getURI())) {
    		element.setNamespace(newNS);
        }
        
        List<Element> elements = element.getChildren();
        for(Element currEl : elements) {
            modifyNamespace(currEl, oldNS, newNS);
        }
    }

    public static String toSafeXml(String val) {
		return val
		.replace((char)0, ' ')
		.replace((char)1, ' ')
		.replace((char)2, ' ')
		.replace((char)3, ' ')
		.replace((char)4, ' ')
		.replace((char)5, ' ')
		.replace((char)6, ' ')
		.replace((char)7, ' ')
		.replace((char)8, ' ')
		.replace((char)11, ' ')
		.replace((char)12, ' ')
		.replace((char)14, ' ')
		.replace((char)15, ' ')
		.replace((char)16, ' ')
		.replace((char)17, ' ')
		.replace((char)18, ' ')
		.replace((char)19, ' ')
		.replace((char)20, ' ')
		.replace((char)21, ' ')
		.replace((char)22, ' ')
		.replace((char)23, ' ')
		.replace((char)24, ' ')
		.replace((char)25, ' ')
		.replace((char)26, ' ')
		.replace((char)27, ' ')
		.replace((char)28, ' ')
		.replace((char)29, ' ')
		.replace((char)30, ' ')
		.replace((char)31, ' ')
		.replace("&", "&amp;")
		.replace("<", "&lt;")
		.replace(">", "&gt;");
	}
}
